<?php
/******************************************************
 * تقرير الإيراد العام (Confirmed Income)
 * - فلترة بالتاريخ (من/إلى)
 * - عرض عمود "الإجمالي بدون الضريبة" من amountwvat
 *   مع بديل احتياطي يحسب من totalamountwvat بنسبة 15%
 * - عرض عمود "الإجمالي مع الضريبة" من totalamountwvat
 * - ملخص (عدد الفواتير / مجموع بدون الضريبة / مجموع مع الضريبة)
 * - تصدير CSV مع نفس الأعمدة
 ******************************************************/

mb_internal_encoding('UTF-8');
date_default_timezone_set('Asia/Riyadh');
error_reporting(E_ALL);
ini_set('display_errors', 1);

require __DIR__ . '/conn.php'; // يجب أن يعرّف $conn كائن mysqli
if (method_exists($conn, 'set_charset')) $conn->set_charset('utf8mb4');

/* ===== إعدادات عامة ===== */
$DEFAULT_VAT_RATE = 0.15; // نسبة ضريبة افتراضية إذا كان amountwvat غير متوفر/غير رقمي
// نحول النسبة لصيغة نصية مناسبة للحقن في SQL بدون sprintf لتفادي Placeholders
$DEFAULT_VAT_RATE_SQL = rtrim(rtrim(number_format((float)$DEFAULT_VAT_RATE, 6, '.', ''), '0'), '.'); // مثال: 0.15

/* 1) التقاط التواريخ والتحقق منها */
$from_date = isset($_GET['from_date']) ? trim($_GET['from_date']) : date('Y-m-d');
$to_date   = isset($_GET['to_date'])   ? trim($_GET['to_date'])   : date('Y-m-d');

$re = '/^\d{4}-\d{2}-\d{2}$/';
if (!preg_match($re, $from_date)) $from_date = date('Y-m-d');
if (!preg_match($re, $to_date))   $to_date   = $from_date;

/* 2) تصحيح المدى */
$fromObj = new DateTime($from_date);
$toObj   = new DateTime($to_date);
if ($toObj < $fromObj) { $tmp = $fromObj; $fromObj = $toObj; $toObj = $tmp; }

$from_dt = $fromObj->format('Y-m-d') . ' 00:00:00';
$to_dt   = $toObj->format('Y-m-d')   . ' 23:59:59';

/* 3) تعبيرات SQL للحسابات:
   - total_without_vat: نقرأ amountwvat إذا كان رقمياً، وإلا نحسب من totalamountwvat / (1 + RATE)
   - total_with_vat  : من totalamountwvat مباشرة
*/
$sql_total_without_vat = "
CAST(
  CASE
    WHEN COALESCE(NULLIF(amountwvat,''),'') REGEXP '^[0-9]+(\\.[0-9]+)?$'
      THEN CAST(amountwvat AS DECIMAL(18,4))
    ELSE
      CAST(totalamountwvat AS DECIMAL(18,4)) / (1 + {$DEFAULT_VAT_RATE_SQL})
  END
AS DECIMAL(18,2))";

$sql_total_with_vat = "CAST(totalamountwvat AS DECIMAL(18,2))";

/* 4) استعلام الصفوف */
$sql_rows = "
  SELECT 
      idinvoice,
      customername,
      servicename,
      {$sql_total_without_vat} AS total_without_vat,
      {$sql_total_with_vat}    AS total_with_vat,
      timeadded
  FROM qrcode
  WHERE COALESCE(LOWER(incomtype),'') = 'confirmed'
    AND timeadded BETWEEN ? AND ?
  ORDER BY timeadded ASC, idinvoice ASC
";

/* 5) وضع التصدير CSV */
if (isset($_GET['export']) && $_GET['export'] === '1') {
    $stmt = $conn->prepare($sql_rows);
    $stmt->bind_param('ss', $from_dt, $to_dt);
    $stmt->execute();
    $res = $stmt->get_result();

    $filename = sprintf(
        'confirmed-income-%s_%s.csv',
        $fromObj->format('Y-m-d'),
        $toObj->format('Y-m-d')
    );

    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="'.$filename.'"');
    header('Pragma: no-cache');
    header('Expires: 0');

    // BOM لعرض العربية بشكل صحيح في Excel
    echo "\xEF\xBB\xBF";

    $out = fopen('php://output', 'w');
    // العناوين
    fputcsv($out, ['رقم الفاتورة','اسم العميل','نوع الخدمة','الإجمالي بدون الضريبة','الإجمالي مع الضريبة','تاريخ الفاتورة']);

    // البيانات
    while ($r = $res->fetch_assoc()) {
        fputcsv($out, [
            $r['idinvoice'],
            $r['customername'],
            $r['servicename'],
            number_format((float)$r['total_without_vat'], 2, '.', ''),
            number_format((float)$r['total_with_vat'], 2, '.', ''),
            $r['timeadded'],
        ]);
    }
    fclose($out);
    exit;
}

/* 6) وضع العرض في الصفحة */
$stmt_rows = $conn->prepare($sql_rows);
$stmt_rows->bind_param('ss', $from_dt, $to_dt);
$stmt_rows->execute();
$res_rows = $stmt_rows->get_result();

/* 7) ملخص */
$sql_sum = "
  SELECT 
      COUNT(*) AS cnt,
      COALESCE(SUM({$sql_total_without_vat}),0) AS total_without_vat_sum,
      COALESCE(SUM({$sql_total_with_vat}),0)    AS total_with_vat_sum
  FROM qrcode
  WHERE COALESCE(LOWER(incomtype),'') = 'confirmed'
    AND timeadded BETWEEN ? AND ?
";
$stmt_sum = $conn->prepare($sql_sum);
$stmt_sum->bind_param('ss', $from_dt, $to_dt);
$stmt_sum->execute();
$summary = $stmt_sum->get_result()->fetch_assoc();

$invoice_count       = (int)$summary['cnt'];
$total_without_sum   = (float)$summary['total_without_vat_sum'];
$total_with_vat_sum  = (float)$summary['total_with_vat_sum'];

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>تقرير الإيراد العام </title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{background:#f8f9fa}
    .card-summary .stat{font-size:1.05rem}
    .amount{font-weight:600}
    .table thead th{white-space:nowrap}
  </style>
</head>
<body>
<div class="container py-4">

  <div class="d-flex align-items-center justify-content-between mb-3">
    <h3 class="m-0">تقرير الإيراد العام</h3>
  </div>

  <!-- نموذج الفلترة + زر التصدير -->
  <form method="get" class="card shadow-sm mb-3">
    <div class="card-body">
      <div class="row g-3 align-items-end">
        <div class="col-md-4">
          <label class="form-label">من تاريخ</label>
          <input type="date" name="from_date" value="<?= htmlspecialchars($fromObj->format('Y-m-d')) ?>" class="form-control" required>
        </div>
        <div class="col-md-4">
          <label class="form-label">إلى تاريخ</label>
          <input type="date" name="to_date" value="<?= htmlspecialchars($toObj->format('Y-m-d')) ?>" class="form-control" required>
        </div>
        <div class="col-md-4 d-flex gap-2">
          <button type="submit" class="btn btn-primary flex-fill">بحث</button>
          <!-- رابط تصدير يحتفظ بنفس نطاق التواريخ -->
          <a class="btn btn-success flex-fill"
             href="?from_date=<?= urlencode($fromObj->format('Y-m-d')) ?>&to_date=<?= urlencode($toObj->format('Y-m-d')) ?>&export=1">
             تصدير CSV
          </a>
          <a class="btn btn-warning"
   href="?from_date=<?= urlencode($fromObj->format('Y-m-d')) ?>&to_date=<?= urlencode($toObj->format('Y-m-d')) ?>&export=push_odoo">
   إرسال إلى أودو
</a>

        </div>
      </div>
    </div>
  </form>

  <!-- الملخص -->
  <div class="card card-summary shadow-sm mb-3">
    <div class="card-body">
      <div class="row text-center">
        <div class="col-md-3 mb-2 mb-md-0">
          <div class="stat text-muted">الفترة المحددة</div>
          <div><?= htmlspecialchars($fromObj->format('Y-m-d')) ?> — <?= htmlspecialchars($toObj->format('Y-m-d')) ?></div>
        </div>
        <div class="col-md-3 mb-2 mb-md-0">
          <div class="stat text-muted">عدد الفواتير المؤكدة</div>
          <div class="fs-5"><?= number_format($invoice_count) ?></div>
        </div>
        <div class="col-md-3 mb-2 mb-md-0">
          <div class="stat text-muted">الإجمالي (بدون الضريبة)</div>
          <div class="fs-5 amount"><?= number_format($total_without_sum, 2) ?> ريال</div>
        </div>
        <div class="col-md-3">
          <div class="stat text-muted">الإجمالي (شامل الضريبة)</div>
          <div class="fs-5 amount"><?= number_format($total_with_vat_sum, 2) ?> ريال</div>
        </div>
      </div>
    </div>
  </div>

  <!-- النتائج -->
  <div class="card shadow-sm">
    <div class="card-body">
      <div class="table-responsive">
        <table class="table table-bordered table-hover align-middle">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>رقم الفاتورة</th>
              <th>اسم العميل</th>
              <th>نوع الخدمة</th>
              <th>الإجمالي بدون الضريبة</th>
              <th>الإجمالي مع الضريبة</th>
              <th>تاريخ الفاتورة</th>
            </tr>
          </thead>
          <tbody>
          <?php
          $rownum = 0;
          if ($res_rows && $res_rows->num_rows > 0):
            while ($r = $res_rows->fetch_assoc()):
              $rownum++;
          ?>
              <tr>
                <td><?= $rownum ?></td>
                <td><?= htmlspecialchars($r['idinvoice']) ?></td>
                <td><?= htmlspecialchars($r['customername']) ?></td>
                <td><?= htmlspecialchars($r['servicename']) ?></td>
                <td class="amount"><?= number_format((float)$r['total_without_vat'], 2) ?> ريال</td>
                <td class="amount"><?= number_format((float)$r['total_with_vat'], 2) ?> ريال</td>
                <td><?= htmlspecialchars($r['timeadded']) ?></td>
              </tr>
          <?php
            endwhile;
          else:
          ?>
              <tr>
                <td colspan="7" class="text-center text-muted">لا توجد فواتير مؤكدة ضمن الفترة المحددة</td>
              </tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

</div>
</body>
</html>
