<?php
/**
 * استيراد خدمات التأمين من ملف CSV فقط إلى جدول insuranceservices
 * - بدون أي مكتبات خارجية.
 * - FACTOR يتم تخزينه كقيمة رقمية (0/1).
 */

mb_internal_encoding('UTF-8');
date_default_timezone_set('Asia/Riyadh');
ini_set('display_errors', 1);
error_reporting(E_ALL);
set_time_limit(0);

// ===== الاتصال بقاعدة البيانات =====
$have_external_conn = false;
if (file_exists(__DIR__ . '/conn.php')) {
    include __DIR__ . '/conn.php';
    if (isset($conn) && $conn instanceof mysqli) {
        $have_external_conn = true;
    }
}
if (!$have_external_conn) {
    // عدّل هذه القيم إذا لم تستخدم conn.php
    $db_host = 'localhost';
    $db_user = 'acesaudi_site';
    $db_pass = 'tabfyv-6xiFry-noczak';
    $db_name = 'acesaudi_usr';
    $db_port = 3306;
    $conn = new mysqli($db_host, $db_user, $db_pass, $db_name, $db_port);
}
if ($conn->connect_error) {
    die('<h3 style="color:#b00">فشل الاتصال بقاعدة البيانات: ' . htmlspecialchars($conn->connect_error) . '</h3>');
}
$conn->set_charset('utf8mb4');

// ===== إعدادات =====
$ALLOWED_EXTS = ['csv'];
$MODE = (isset($_POST['mode']) && strtolower($_POST['mode']) === 'upsert') ? 'upsert' : 'insert';

// ===== دوال مساعدة =====
function strip_bom($s) {
    if (is_string($s) && substr($s, 0, 3) === "\xEF\xBB\xBF") return substr($s, 3);
    return $s;
}
function norm_key($s) {
    $s = trim((string)$s);
    $s = str_replace(["\xC2\xA0"], ' ', $s); // non-breaking space
    $s = strip_bom($s);
    $s = mb_strtolower($s, 'UTF-8');
    $s = preg_replace('/\s+/u', '_', $s);
    $s = str_replace(['-'], '_', $s);
    return $s;
}
function arabic_to_english_digits($str) {
    $ar = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩','٫','٬','،'];
    $en = ['0','1','2','3','4','5','6','7','8','9','.','',','];
    return str_replace($ar, $en, (string)$str);
}
function clean_price($v) {
    $v = arabic_to_english_digits($v);
    $v = trim((string)$v);
    $v = str_replace([',', ' '], '', $v);      // إزالة فواصل الآلاف
    $v = preg_replace('/[^0-9\.\-]/', '', $v); // احتفظ بالأرقام والنقطة والسالب
    return $v; // سيتم تمريره كنص مع NULLIF
}
function normalize_factor_to_bit($v) {
    // يُعيد 0 أو 1 فقط
    if ($v === null) return 1; // افتراضي = 1
    $v = is_string($v) ? trim($v) : $v;
    $v = (string)$v;
    // إزالة رموز خفية ومسافات
    $v = preg_replace('/[\x{200B}-\x{200D}\x{FEFF}]/u', '', $v);
    $vl = mb_strtolower($v, 'UTF-8');
    $vl_ns = preg_replace('/\s+/u', '', $vl);

    $truthy = ['1','on','yes','true','active','y','t','نشط','مفعل','مفعّل','enabled'];
    $falsy  = ['0','off','no','false','inactive','n','f','غيرنشط','غير_نشط','معطل','موقوف','disabled'];

    if ($v === '' ) return 1;
    if (in_array($vl, $truthy, true) || in_array($vl_ns, $truthy, true)) return 1;
    if (in_array($vl, $falsy,  true) || in_array($vl_ns, $falsy,  true)) return 0;
    if (ctype_digit($v)) return ((int)$v) ? 1 : 0;

    return 1;
}

// أسماء الأعمدة المقبولة (مع Aliases مرنة + إصلاح الخطأ المطبعي الشائع)
$aliases = [
    'insurancecompanyid'   => ['insurancecompanyid','company_id','insurance_company_id'],
    'insurancecopmanyname' => ['insurancecopmanyname','company_name','insurance_company_name','insurancecopmanyname'], // دعم الخطأ المطبعي
    'service_code'         => ['service_code','servicecode','code'],
    'non_standard_code'    => ['non_standard_code','nonstandardcode','non_standard'],
    'service_type'         => ['service_type','servicetype','type'],
    'service_name_ar'      => ['service_name_ar','servicenamear','arabic_name','name_ar'],
    'service_name_en'      => ['service_name_en','servicenameen','english_name','name_en'],
    'non_standard_desc'    => ['non_standard_desc','nonstandarddesc','desc','description'],
    'unit_price'           => ['unit_price','unitprice','price','amount'],
    'specialty'            => ['specialty','speciality','department'],
    'factor'               => ['factor','status']
];
$required_min = ['insurancecompanyid','insurancecopmanyname','service_code'];

// ===== صفحة الرفع (GET) =====
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<title>استيراد خدمات التأمين (CSV فقط)</title>
<style>
body{font-family:Tahoma,Arial;margin:30px}
.box{border:1px solid #ddd;padding:20px;border-radius:8px;max-width:700px}
.hint{color:#555}.note{background:#f8f8f8;padding:10px;border-radius:6px;margin-top:10px}
</style>
</head>
<body>
<div class="box">
    <h2>استيراد خدمات التأمين إلى جدول <code>insuranceservices</code> (CSV فقط)</h2>
    <form method="post" enctype="multipart/form-data">
        <label>اختر ملف CSV:
            <input type="file" name="csv" accept=".csv" required>
        </label>
        <br><br>
        <label>وضع التشغيل:
            <select name="mode">
                <option value="insert">إدراج فقط</option>
                <option value="upsert">تحديث إن وجد (insurancecompanyid + SERVICE_CODE)</option>
            </select>
        </label>
        <br><br>
        <button type="submit">بدء الاستيراد</button>
    </form>

    <div class="note">
        <b>يجب أن يحتوي الصف الأول على عناوين الأعمدة:</b><br>
        insurancecompanyid, insurancecopmanyname, SERVICE_CODE, NON_STANDARD_CODE, SERVICE_TYPE, service_name_ar, service_name_en, NON_STANDARD_DESC, UNIT_PRICE, specialty, FACTOR
        <br>الحقول الدنيا المطلوبة: <b>insurancecompanyid, insurancecopmanyname, SERVICE_CODE</b>
        <br><b>FACTOR</b>: اكتب 1 أو 0 (الافتراضي 1 إذا تُركت فارغة).
    </div>
    <div class="note hint">
        لمنع التكرار على مستوى القاعدة نفّذ مرة واحدة:<br>
        <code>ALTER TABLE insuranceservices ADD UNIQUE KEY uniq_comp_code (insurancecompanyid, SERVICE_CODE);</code>
    </div>
</div>
</body>
</html>
<?php
exit;
}

// ===== معالجة الرفع (POST) =====
if (!isset($_FILES['csv']) || $_FILES['csv']['error'] !== UPLOAD_ERR_OK) {
    die('<h3 style="color:#b00">لم يتم رفع ملف CSV بشكل صحيح.</h3>');
}
$ext = strtolower(pathinfo($_FILES['csv']['name'], PATHINFO_EXTENSION));
if (!in_array($ext, $ALLOWED_EXTS)) {
    die('<h3 style="color:#b00">الامتداد غير مدعوم. المسموح: CSV فقط.</h3>');
}
$csvPath = $_FILES['csv']['tmp_name'];

// تخمين الفاصل (، أو ;) من أول 2KB
$firstBytes = file_get_contents($csvPath, false, null, 0, 2048);
$commaCount = substr_count($firstBytes, ',');
$semiCount  = substr_count($firstBytes, ';');
$delimiter  = ($semiCount > $commaCount) ? ';' : ','; // الافتراضي الفاصلة ,

// فتح الملف
$h = fopen($csvPath, 'r');
if (!$h) { die('<h3 style="color:#b00">تعذّر فتح ملف CSV.</h3>'); }

// قراءة العناوين
$header = fgetcsv($h, 0, $delimiter);
if (!$header || count($header) === 0) { fclose($h); die('<h3 style="color:#b00">ملف CSV لا يحتوي على صف عناوين.</h3>'); }
foreach ($header as &$t) { $t = strip_bom($t); }
unset($t);

// بناء خريطة الأعمدة: index => canonical_key
$colMap = []; // مثال: 0 => 'insurancecompanyid'
for ($i = 0; $i < count($header); $i++) {
    $nk = norm_key($header[$i]);
    $matched = null;
    foreach ($aliases as $canon => $alts) {
        if (in_array($nk, $alts, true) || $nk === $canon) { $matched = $canon; break; }
    }
    if ($matched) $colMap[$i] = $matched;
}

// تحقّق من الأعمدة الدنيا
$found = array_values(array_unique(array_values($colMap)));
$missing = array_diff($required_min, $found);
if ($missing) {
    fclose($h);
    die('<h3 style="color:#b00">الأعمدة المطلوبة مفقودة في CSV: ' . htmlspecialchars(implode(', ', $missing)) . '</h3>');
}

// تحضير العبارات
$sqlInsert = "INSERT INTO insuranceservices
(insurancecompanyid, insurancecopmanyname, SERVICE_CODE, NON_STANDARD_CODE, SERVICE_TYPE, service_name_ar, service_name_en, NON_STANDARD_DESC, UNIT_PRICE, specialty, FACTOR)
VALUES (?,?,?,?,?,?,?,?, NULLIF(?,''), ?, ?)";
$stmtIns = $conn->prepare($sqlInsert);
if (!$stmtIns) { fclose($h); die('<h3 style="color:#b00">خطأ تجهيز INSERT: ' . htmlspecialchars($conn->error) . '</h3>'); }

$sqlUpdate = "UPDATE insuranceservices SET
insurancecopmanyname=?, NON_STANDARD_CODE=?, SERVICE_TYPE=?, service_name_ar=?, service_name_en=?, NON_STANDARD_DESC=?, UNIT_PRICE=NULLIF(?,''), specialty=?, FACTOR=?
WHERE insurancecompanyid=? AND SERVICE_CODE=?";
$stmtUpd = $conn->prepare($sqlUpdate);
if (!$stmtUpd) { fclose($h); die('<h3 style="color:#b00">خطأ تجهيز UPDATE: ' . htmlspecialchars($conn->error) . '</h3>'); }

$sqlCheck = "SELECT service_id FROM insuranceservices WHERE insurancecompanyid=? AND SERVICE_CODE=? LIMIT 1";
$stmtCheck = $conn->prepare($sqlCheck);
if (!$stmtCheck) { fclose($h); die('<h3 style="color:#b00">خطأ تجهيز SELECT: ' . htmlspecialchars($conn->error) . '</h3>'); }

// تنفيذ داخل معاملة
$conn->begin_transaction();
$inserted = 0; $updated = 0; $skipped = 0;
$errors = [];
$lineNo = 1; // لأننا قرأنا العناوين

try {
    while (($row = fgetcsv($h, 0, $delimiter)) !== false) {
        $lineNo++;

        // تجاهل الصفوف الفارغة
        $nonEmpty = false;
        foreach ($row as $cell) { if (trim((string)$cell) !== '') { $nonEmpty = true; break; } }
        if (!$nonEmpty) { continue; }

        // اجلب القيم بحسب colMap
        $data = [
            'insurancecompanyid'   => null,
            'insurancecopmanyname' => null,
            'service_code'         => null,
            'non_standard_code'    => null,
            'service_type'         => null,
            'service_name_ar'      => null,
            'service_name_en'      => null,
            'non_standard_desc'    => null,
            'unit_price'           => null,
            'specialty'            => null,
            'factor'               => null,
        ];
        foreach ($colMap as $idx => $canon) {
            $val = isset($row[$idx]) ? trim((string)$row[$idx]) : null;
            if ($canon === 'unit_price') $val = clean_price($val);
            $data[$canon] = $val;
        }

        // تحقق من الحد الأدنى
        if (!$data['insurancecompanyid'] || !$data['insurancecopmanyname'] || !$data['service_code']) {
            $skipped++;
            $errors[] = "سطر $lineNo: بيانات ناقصة (insurancecompanyid/insurancecopmanyname/SERVICE_CODE).";
            continue;
        }

        // عامل الحالة كرقم 0/1
        $factor = normalize_factor_to_bit($data['factor'] ?? '');

        if ($MODE === 'upsert') {
            $icid  = $data['insurancecompanyid'];
            $scode = $data['service_code'];

            $stmtCheck->bind_param('ss', $icid, $scode);
            $stmtCheck->execute();
            $stmtCheck->store_result();

            if ($stmtCheck->num_rows > 0) {
                // تحديث
                $stmtUpd->bind_param(
                    'ssssssssiss', // 8 s + i + 2 s
                    $data['insurancecopmanyname'],
                    $data['non_standard_code'],
                    $data['service_type'],
                    $data['service_name_ar'],
                    $data['service_name_en'],
                    $data['non_standard_desc'],
                    $data['unit_price'],
                    $data['specialty'],
                    $factor,   // i
                    $icid,     // s
                    $scode     // s
                );
                if (!$stmtUpd->execute()) {
                    $skipped++;
                    $errors[] = "سطر $lineNo: فشل التحديث - " . $stmtUpd->error;
                } else {
                    $updated++;
                }
            } else {
                // إدراج
                $stmtIns->bind_param(
                    'ssssssssssi', // 10 s + i
                    $data['insurancecompanyid'],
                    $data['insurancecopmanyname'],
                    $data['service_code'],
                    $data['non_standard_code'],
                    $data['service_type'],
                    $data['service_name_ar'],
                    $data['service_name_en'],
                    $data['non_standard_desc'],
                    $data['unit_price'],
                    $data['specialty'],
                    $factor    // i
                );
                if (!$stmtIns->execute()) {
                    $skipped++;
                    $errors[] = "سطر $lineNo: فشل الإدراج - " . $stmtIns->error;
                } else {
                    $inserted++;
                }
            }
        } else {
            // إدراج فقط
            $stmtIns->bind_param(
                'ssssssssssi', // 10 s + i
                $data['insurancecompanyid'],
                $data['insurancecopmanyname'],
                $data['service_code'],
                $data['non_standard_code'],
                $data['service_type'],
                $data['service_name_ar'],
                $data['service_name_en'],
                $data['non_standard_desc'],
                $data['unit_price'],
                $data['specialty'],
                $factor
            );
            if (!$stmtIns->execute()) {
                $skipped++;
                $errors[] = "سطر $lineNo: فشل الإدراج - " . $stmtIns->error;
            } else {
                $inserted++;
            }
        }
    }
    fclose($h);
    $conn->commit();
} catch (Throwable $e) {
    if (is_resource($h)) fclose($h);
    $conn->rollback();
    die('<h3 style="color:#b00">تم التراجع عن العملية بسبب خطأ: ' . htmlspecialchars($e->getMessage()) . '</h3>');
}

// ===== إخراج الملخّص =====
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<title>نتيجة الاستيراد (CSV)</title>
<style>
body{font-family:Tahoma,Arial;margin:30px}
.ok{color:green}.err{color:#b00}.hint{color:#555}
.box{border:1px solid #ddd;padding:20px;border-radius:8px;max-width:900px}
</style>
</head>
<body>
<div class="box">
    <h2>نتيجة الاستيراد (وضع: <?= htmlspecialchars($MODE) ?>)</h2>
    <ul>
        <li class="ok">تم الإدراج: <?= (int)$inserted ?></li>
        <li class="ok">تم التحديث: <?= (int)$updated ?></li>
        <li class="err">تم التخطي: <?= (int)$skipped ?></li>
    </ul>

    <?php if (!empty($errors)): ?>
        <h3 class="err">تفاصيل السطور المتخطّاة (أول 50 فقط):</h3>
        <ol>
            <?php foreach (array_slice($errors, 0, 50) as $msg): ?>
                <li class="err"><?= htmlspecialchars($msg) ?></li>
            <?php endforeach; ?>
        </ol>
    <?php endif; ?>

    <p class="hint">
        للتأكّد من عدم تكرار نفس الخدمة لنفس الشركة، أضف الفهرس الفريد مرّة واحدة:<br>
        <code>ALTER TABLE insuranceservices ADD UNIQUE KEY uniq_comp_code (insurancecompanyid, SERVICE_CODE);</code>
    </p>

    <p><a href="<?= htmlspecialchars($_SERVER['PHP_SELF']) ?>">رجوع لصفحة الرفع</a></p>
</div>
</body>
</html>
